% The MATLAB code generates the pattern of the PEC layer on the metasurface
% Reference:  Menglin L.N. Chen, Li Jun Jiang, and Wei E.I. Sha, 
% "Quasi-Continuous Metasurfaces for Orbital Angular Momentum Generation," 
% IEEE Antennas and Wireless Propagation Letters, 2019, Accepted for Publication.  
% http://doi.org/10.1109/LAWP.2019.2894772
% Developer: Menglin L. N. Chen, EEE Dept. the University of Hong Kong, Hong Kong
% email: menglin@connect.hku.hk

% polar mesh
nr=1000;
ntheta=nr;
rsize=100e-3;  % unit: m
rstart=5e-3;
thetas=0;
thetae=2*pi;
r=linspace(rstart,rsize,nr);
theta=linspace(thetas,thetae,ntheta);
deltar=(rsize-rstart)/(nr-1);
deltat=(thetae-thetas)/(ntheta-1);
[rm,thm]=meshgrid(r,theta);
 
% polar to cartesian
xm=rm.*cos(thm);
ym=rm.*sin(thm);
 
% initial parameter
lambda=3e8/6.2e9;
k=2*pi/lambda;
ang=0*pi/180;  % directivity, theta0
kt=sin(ang)*k;
ang0=0*pi/180;  % directivity, phi0
ky=sin(ang0).*kt;
kx=cos(ang0).*kt;
 
% parameters in Eq.(2) in the main text
a=kx/2;
b=ky/2;
m=0.5;  % topological charge
phi0=pi/2;  % phase constant, D
theta_des=m.*thm+a.*xm +b.*ym+phi0;  % desired orientation of metal strips

% kg along r and phi
k0=exp(a.*ym-b.*xm)./(rm).^(m);
kg_r=k0.*cos(theta_des-thm);
kg_th=k0.*sin(theta_des-thm);

% doing integral
rr{1}=0.5;  % initial value
tt{1}=0;
for nn=2:nr
    rr{nn}=rr{nn-1}+trapz([kg_r(1,nn-1),kg_r(1,nn)])*deltar;
end
 
for mm=1:ntheta
    ntheta-mm
    for nn=1:nr
        phi(mm,nn)=rr{nn}+trapz([kg_th(1:mm,nn)])*rm(mm,nn)*deltat;
    end
end
 
figure;
pcolor(xm,ym,phi);
shading interp;
colormap jet
colorbar
title('g')
 
% generating Lee-type binary grating
q=0.5;  % duty circle 
scale=300;  % scaling factor
phi=phi.*scale;

diff=cos(phi)-cos(pi*q);
phi_g=zeros(nr,nr);
 
for m=1:nr
    for n=1:nr
        if diff(m,n)>=0
            phi_g(m,n)=1;
        end
    end
end
 
figure;
pcolor(xm,ym,phi_g);
shading interp;
colormap jet
colorbar
title('binary grating')
